package gov.va.vamf.scheduling.direct.datalayer.clinic;

import com.agilex.healthcare.directscheduling.dao.DAOConstants;
import com.agilex.healthcare.directscheduling.dao.PatientRelationshipDao;
import com.agilex.healthcare.directscheduling.dataservice.DSFacilityProviderDataService;
import com.agilex.healthcare.directscheduling.domain.FacilityMemberClinic;
import com.agilex.healthcare.veteranappointment.dataservice.DisabledFeatureService;
import gov.va.vamf.scheduling.direct.datalayer.facility.CustomFriendlyTextService;
import gov.va.vamf.scheduling.direct.domain.CdwClinic;
import gov.va.vamf.scheduling.direct.domain.CdwClinics;
import gov.va.vamf.scheduling.direct.domain.ClinicalService;
import gov.va.vamf.scheduling.direct.domain.CoreSettings;
import gov.va.vamf.scheduling.direct.domain.CustomFriendlyText;
import gov.va.vamf.scheduling.direct.domain.StopCode;
import gov.va.vamf.scheduling.direct.domain.StopCodes;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Component
public class ClinicDataService {

    private static String TH_FEATURE_SET = "TH_FEATURE_SET";
    private static String VIDEO_VISIT_STOP_CODE = "179";

    @Resource(name = "clinicDaoImpl")
    ClinicDaoImpl clinicDao;

    @Resource(name = "patientRelationshipDaoClinicImpl")
    private PatientRelationshipDao patientRelationshipClinicDao;

    @Resource
    private DSFacilityProviderDataService dsFacilityProviderDataService;

    @Resource
    CustomFriendlyTextService customFriendlyTextService;

    @Resource
    DisabledFeatureService disabledFeatureService;

    public CdwClinics retrieveDirectSchedulingClinicsByStopCodes(String siteCode, StopCodes stopCodes) {
        return clinicDao.retrieveDirectSchedulingClinicsByStopCodes(siteCode, stopCodes);
    }

    public CdwClinics retrieveDisplayToPatientClinics(String siteCode) {
        return clinicDao.retrieveDisplayToPatientClinics(siteCode);
    }

    public CdwClinics retrieveDisplayToPatientClinics(String siteCode, Collection<String> clinicIENs) {
        return clinicDao.retrieveDisplayToPatientClinics(siteCode, clinicIENs);
    }

    public CdwClinics retrieveClinicsByStopCodes(String siteCode, StopCodes stopCodes) {
        return clinicDao.retrieveClinicsByStopCodes(siteCode, stopCodes);
    }

    public Collection<CdwClinic> retrieveDisplayToPatientClinicsByStopCode(
            final String patientIcn,
            final String threeDigitCode,
            final String institutionCode,
            final String clinicalServiceId
    ) {
        final List<StopCode> excludedStopCodes = new ArrayList<>(1);

        if (disabledFeatureService.containsDisabledFeature(TH_FEATURE_SET)) {
            // Exclude any clinic with a video visit secondary stop code
            excludedStopCodes.add(new StopCode(null, VIDEO_VISIT_STOP_CODE));
        }

        if (clinicalServiceId.equalsIgnoreCase(CoreSettings.PRIMARY_CARE_ID)) {
            return retrieveDisplayToPatientPrimaryCareClinics(patientIcn, threeDigitCode, institutionCode, excludedStopCodes);
        }
        else {
            return clinicDao.retrieveDisplayToPatientClinicsByStopCode(institutionCode, clinicalServiceId, excludedStopCodes);
        }
    }

    // TODO: move this somewhere more appropriate
    public CustomFriendlyText retrieveChildFacilityCustomFriendlyTexts(String institutionCode) {
        return customFriendlyTextService.fetchByInstitutionCode(institutionCode);
    }

    private Collection<CdwClinic> retrieveDisplayToPatientPrimaryCareClinics(
            final String patientIcn,
            final String threeDigitCode,
            final String institutionCode,
            final Collection<StopCode> excludedStopCodes
    ) {
        final StopCodes stopCodes =
                dsFacilityProviderDataService.createStopCodesFromPrimary(CoreSettings.PRIMARY_CARE_ID);
        final String stopCodesString = ClinicalService.getStopCodesAsString(stopCodes);

        Collection<FacilityMemberClinic> clinics = patientRelationshipClinicDao.getPatientData(new HashMap<String, Object>() {{
            put(DAOConstants.PATIENT_ICN, patientIcn);
            put(DAOConstants.STOP_CODE_PAIRS, stopCodesString);
        }});

        final List<CdwClinic> cdwClinics = createCdwClinicsFromFacilityMemberClinics(
                filterFacilityMemberClinics(clinics, threeDigitCode, institutionCode, excludedStopCodes)
        ).collect(Collectors.toList());

        return cdwClinics;
    }

    private Stream<CdwClinic> createCdwClinicsFromFacilityMemberClinics(Stream<FacilityMemberClinic> facilityMemberClinicStream) {
        return facilityMemberClinicStream.map(new Function<FacilityMemberClinic, CdwClinic>() {
            @Override
            public CdwClinic apply(FacilityMemberClinic facilityMemberClinic) {
                final CdwClinic cdwClinic = new CdwClinic();
                cdwClinic.setClinicId(facilityMemberClinic.getLocationIEN());
                cdwClinic.setClinicFriendlyLocationName(facilityMemberClinic.getFriendlyLocationName());
                cdwClinic.setClinicName(facilityMemberClinic.getLocationName());
                cdwClinic.setDirectSchedulingFlag("Y");
                cdwClinic.setDisplayToPatientFlag("Y");
                cdwClinic.setInstitutionCode(facilityMemberClinic.getInstitutionCode());
                cdwClinic.setPrimaryStopCode(facilityMemberClinic.getPrimaryStopCode());
                cdwClinic.setSecondaryStopCode(facilityMemberClinic.getSecondaryStopCode());
                cdwClinic.setInstitutionName(facilityMemberClinic.getInstitutionName());
                cdwClinic.setSiteCode(facilityMemberClinic.getFacilityId());

                return cdwClinic;
            }
        });
    }
    
    private Stream<FacilityMemberClinic> filterFacilityMemberClinics(
            final Collection<FacilityMemberClinic> clinics,
            final String threeDigitCode,
            final String institutionCode,
            final Collection<StopCode> excludedStopCodes
    ) {
        if (clinics == null) {
            return Stream.empty();
        }

        return clinics
                .stream()
                .filter(new Predicate<FacilityMemberClinic>() {
                    @Override
                    public boolean test(final FacilityMemberClinic facilityMemberClinic) {
                        if (institutionCode != null && !institutionCode.equals(facilityMemberClinic.getInstitutionCode())) {
                            return false;
                        }

                        if (!threeDigitCode.equals(facilityMemberClinic.getFacilityId())) {
                            return false;
                        }

                        final String primaryStopCode = facilityMemberClinic.getPrimaryStopCode();
                        final String secondaryStopCode = facilityMemberClinic.getSecondaryStopCode();

                        boolean clinicStopCodeIsExcluded =
                                StopCode.matchAny(excludedStopCodes, new StopCode(primaryStopCode, secondaryStopCode));

                        return !clinicStopCodeIsExcluded;
                    }
                });
    }
}
